/*

Modelspoor stack modules
www.stackmodules.nl
(c) Ed den Ouden 2023

Library:      LocoNet Github
Bestand:      BB LocoClient-T LM311 - Outputs wissels
Functie:      switch voor 8 wissels
Board:        Nano V3 328p
Kanalen:      8, voor 8 wissels met elk 2 motorkanalen (16 pinnen)
Configuratie: zie variabelen bij 'Door de gebruiker in te stellen variabelen'

Adressering:  deze LocoClient heeft 16 uitgangen voor 8 wissels, en de nummering begint bij adres 1 (of 17, 25, 33, 41, 49 etc (groepen van 8))

              stel dat het startadres 17 is, de adressering wordt dan bv in Rocrail (adres-poort):

              LN      Adres Poort
              -------------------
              17      5     2
              18      5     3
              19      5     4
              20      6     1
              21      6     2
              22      6     3
              23      6     4
              24      7     1


*/



// Door de gebruiker in te stellen variabelen

// #define SERIALMON  // Comment/uncomment schakelt de seriële monitor uit/aan

int loconetStartadres = 17;  // LocoNet startadres van deze stack module (1-4096), de volgende 8 kanalen nummeren oplopend
int aantalKanalen = 8;       // Aantal wisselkanalen van deze stack module (default = 8)
int aantalPinnen = 16;       // Aantal aan te sturen pinnen, 2 pinnen per wisselkanaal (default is 16)

// Pinmapping basis board LocoNet, pinnen 6 en 8 worden intern gebruikt
int usedPins[] = { 2, 3, 4, 5, 7, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 };

// Einde door de gebruiker in te stellen variabelen



int boardChannel;     // Aan te sturen kanaal op het stack board, range 0-15 (16 kanalen)
#include <LocoNet.h>  // Library
lnMsg *LnPacket;      // Pointer operator



void setup() {

  LocoNet.init();  // Initialiseer library

  // Definieer 16 pinnen van de pinmapping als output
  for (byte i = 0; i < aantalPinnen; i++) {
    pinMode(usedPins[i], OUTPUT);
  }

#ifdef SERIALMON
  Serial.begin(57600);
  Serial.println("Modelspoor Stack Module V2.1 (Nano V3 328p) - Ed den Ouden - 2023");
  Serial.println("LocoClient outputs voor elektromagnetische wissels (www.stackmodules.nl)");
  Serial.println("");
  Serial.print("Startadres: ");
  Serial.println(loconetStartadres);
  Serial.print("Aantal wissels: ");
  Serial.println(aantalKanalen);
  Serial.println("");
  Serial.println("Initialisatie klaar, wacht op instructies van de centrale...");
  Serial.println();
#endif
}



void loop() {

  // Check for any received LocoNet packets
  LnPacket = LocoNet.receive();

  if (LnPacket) {
// First print out the packet in HEX
#ifdef SERIALMON
    Serial.print("RX: ");
#endif
    uint8_t msgLen = getLnMsgSize(LnPacket);
    for (uint8_t x = 0; x < msgLen; x++) {
      uint8_t val = LnPacket->data[x];

// Print a leading 0 if less than 16 to make 2 HEX digits
#ifdef SERIALMON
      if (val < 16)
        Serial.print('0');

      Serial.print(val, HEX);
      Serial.print(' ');
#endif
    }

    // If this packet was not a Switch or Sensor Message then print a new line
    if (!LocoNet.processSwitchSensorMessage(LnPacket)) {
      Serial.println();
    }
  }
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Sensor messages
void notifySensor(uint16_t Address, uint8_t State) {

#ifdef SERIALMON
  Serial.print("Sensor: ");
  Serial.print(Address, DEC);
  Serial.print(" - ");
  Serial.println(State ? "Active" : "Inactive");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Request messages
void notifySwitchRequest(uint16_t Address, uint8_t Output, uint8_t Direction) {

  // Bepaal of het ontvangen adres bij deze stack module hoort en schakel vervolgens de opdracht (aan of uit)
  if (Address >= loconetStartadres && Address <= (loconetStartadres + aantalKanalen )) {

    // Bepaal welke usedPins[boardChannel]) bij deze combi adres/direction hoort (8 paren van 2 pinnen)
    boardChannel = Address - loconetStartadres - 1;
    boardChannel = boardChannel * 2;
    if (Direction) { boardChannel = boardChannel; }       // Aan te sturen pin bij rechtdoor
    if (!Direction) { boardChannel = boardChannel + 1; }  // Aan te sturen pin bij afslaan


#ifdef SERIALMON
    Serial.print("Adres: ");
    Serial.print(Address - 1, DEC);
    Serial.print(" - ");
    Serial.print(Direction ? "Rechtdoor" : "Afbuigen ");
    Serial.print(" - ");
    Serial.print(Output ? "Aan" : "Uit");
    Serial.print(" - ");
    Serial.print(" Board channel: ");
    Serial.println(boardChannel);

#endif



    if (Direction) {  // Rechtdoor, 1e adres
      digitalWrite(usedPins[boardChannel], Output);
    }
    if (!Direction) {  // Afbuigen, 2e adres
      digitalWrite(usedPins[boardChannel], Output);
    }
  }
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Output Report messages
void notifySwitchOutputsReport(uint16_t Address, uint8_t ClosedOutput, uint8_t ThrownOutput) {

#ifdef SERIALMON
  Serial.print("Switch Outputs Report: ");
  Serial.print(Address, DEC);
  Serial.print(": Closed - ");
  Serial.print(ClosedOutput ? "On" : "Off");
  Serial.print(": Thrown - ");
  Serial.println(ThrownOutput ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Sensor Report messages
void notifySwitchReport(uint16_t Address, uint8_t State, uint8_t Sensor) {

#ifdef SERIALMON
  Serial.print("Switch Sensor Report: ");
  Serial.print(Address, DEC);
  Serial.print(':');
  Serial.print(Sensor ? "Switch" : "Aux");
  Serial.print(" - ");
  Serial.println(State ? "Active" : "Inactive");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch State messages
void notifySwitchState(uint16_t Address, uint8_t Output, uint8_t Direction) {

#ifdef SERIALMON
  Serial.print("Switch State: ");
  Serial.print(Address, DEC);
  Serial.print(':');
  Serial.print(Direction ? "Closed" : "Thrown");
  Serial.print(" - ");
  Serial.println(Output ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Power messages
void notifyPower(uint8_t State) {

#ifdef SERIALMON
  Serial.print("Layout Power State: ");
  Serial.println(State ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all MultiSensePower messages
void notifyMultiSensePower(uint8_t BoardID, uint8_t Subdistrict, uint8_t Mode, uint8_t Direction) {

#ifdef SERIALMON
  Serial.print("MultiSensePower: Board ID: ");
  Serial.print(BoardID, DEC);
  Serial.print(" Sub District: ");
  Serial.print(Subdistrict, DEC);
  Serial.print(" Mode: ");
  Serial.print(Mode, DEC);
  Serial.print(" Direction: ");
  Serial.println(Direction, DEC);
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all notifyMultiSenseTransponder messages
void notifyMultiSenseTransponder(uint16_t Address, uint8_t Zone, uint16_t LocoAddress, uint8_t Present) {

#ifdef SERIALMON
  Serial.print("MultiSenseTransponder: Address: ");
  Serial.print(Address, DEC);
  Serial.print(" Zone: ");
  Serial.print(Zone, DEC);
  Serial.print(" Loco Address: ");
  Serial.print(LocoAddress, DEC);
  Serial.print(" Present: ");
  Serial.println(Present, DEC);
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all LongAck messages

void notifyLongAck(uint8_t d1, uint8_t d2) {

#ifdef SERIALMON
  Serial.print("LongACK : Data Byte 1: ");
  Serial.print(d1, DEC);
  Serial.print(" Data Byte 2: ");
  Serial.println(d2, DEC);
#endif
}
